/* Copyright (C) 2014-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "StringGenerator.h"
#include <DiscoveryException.h>
#include <assert.h>

using namespace vncdiscovery;

namespace
{
  VNCDiscoverySDKGenerator* createGenerator(VNCDiscoverySDKUtils *pUtils)
  {
    if(!pUtils)
    {
      throw DiscoveryException(
          "Unable to create Generator, Utils structure is NULL",
          VNCDiscoverySDKErrorInvalidParameter);
    }
    if(!pUtils->createGenerator)
    {
      throw DiscoveryException(
          "Unable to create Generator, the operation is not supported",
          VNCDiscoverySDKErrorNotSupported);
    }

    VNCDiscoverySDKGenerator* pGenerator = NULL;
    VNCDiscoverySDKError err = pUtils->createGenerator(
        pUtils->pSDKInstance, &pGenerator);

    if(err != VNCDiscoverySDKErrorNone)
    {
      assert(!pGenerator);
      if(pGenerator && pUtils->generatorDestroy)
      {
        pUtils->generatorDestroy(pGenerator);
      }
      throw DiscoveryException(
          "Error when trying to create Generator", err);
    }
    else if(!pGenerator)
    {
      // Shouldn't really get into here. If the SDK returned error none, then
      // the Generator should have been created and initialized
      assert(false);
      throw DiscoveryException("Generator was not created",
          VNCDiscoverySDKErrorNotInitialized);
    }
    return pGenerator;
  }
}

StringGenerator::StringGenerator(VNCDiscoverySDKUtils *pUtils)
  : mpUtils(pUtils),
    mpGenerator(NULL)
{
  mpGenerator = createGenerator(pUtils);
  assert(mpGenerator);
}

StringGenerator::~StringGenerator()
{
  if(mpGenerator && mpUtils->generatorDestroy)
  {
    mpUtils->generatorDestroy(mpGenerator);
  }
}

VNCDiscoverySDKError StringGenerator::setStringField(const char* fieldPath,
    const char* value)
{
  if(!mpUtils->generatorSetStringField)
  {
    return VNCDiscoverySDKErrorNotSupported;
  }
  return mpUtils->generatorSetStringField(mpGenerator, fieldPath, value);
}

VNCDiscoverySDKError StringGenerator::setInt32Field(const char* fieldPath,
    vnc_int32_t value)
{
  if(!mpUtils->generatorSetInt32Field)
  {
    return VNCDiscoverySDKErrorNotSupported;
  }
  return mpUtils->generatorSetInt32Field(mpGenerator, fieldPath, value);
}

const char* StringGenerator::generate(VNCDiscoverySDKError* pError)
{
  const char* pResult = NULL;
  VNCDiscoverySDKError err = VNCDiscoverySDKErrorNotSupported;
  if(mpUtils->generatorGenerate)
  {
    err = mpUtils->generatorGenerate(mpGenerator, &pResult);
  }
  if(pError)
  {
    *pError = err;
  }
  return pResult;
}

